
import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Condition.custom;
import static com.xebialabs.restito.semantics.Condition.method;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;

import org.glassfish.grizzly.http.Method;
import org.glassfish.grizzly.http.server.Response;
import org.glassfish.grizzly.http.util.HttpStatus;

import com.xebialabs.restito.semantics.Call;
import com.xebialabs.restito.semantics.Function;
import com.xebialabs.restito.semantics.Predicate;
import com.xebialabs.restito.server.StubServer;

public class BookedAppointmentsMock extends AbstractMock {
    private static String appointmentTemplate;
    private static String appointmentsTemplate;

    public BookedAppointmentsMock() {
        appointmentTemplate  = loadTemplate("appointment-template.txt");
        appointmentsTemplate = loadTemplate("appointments-template.txt");
    }

    @Override
    public void run(StubServer server) {
        Predicate<Call> validBookedAppointmentRequest = new Predicate<Call>() {
            @Override
            public boolean apply(final Call input) {
                return input.getUri().matches(".*/VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/\\d+/patient/EDIPI/.*/booked-appointments");
            }
        };

        Predicate<Call> validBookedAppointmentPostRequest = new Predicate<Call>() {
            @Override
            public boolean apply(final Call input) {
                return input.getUri().matches(".*/VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/\\d+/patient/EDIPI/.*/booked-appointments");
            }
        };

        Predicate<Call> requestToGenerateErrorMessage = new Predicate<Call>() {
            @Override
            public boolean apply(final Call input) {
                return input.getUri().matches(".*/VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/\\d+/patient/EDIPI/0000000001/booked-appointments");
            }
        };

        Function<Response, Response> appointments = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                String uri = input.getRequest().getRequestURI();
                String siteCodeRegex = ".*/site/(\\d+)/patient";
                String patientIdRegex = ".*/EDIPI/(.*)/booked-appointments";

                String host = "https://IP      ";
                String siteCode = matchingRegex(uri, siteCodeRegex);
                String patientId = matchingRegex(uri, patientIdRegex);

                Method method = input.getRequest().getMethod();
                String output = "{\"bookedAppointmentCollections\": []}";

                if (patientId.equals("P011")) {
                    output = createNotCancellableAppointments(host, siteCode, patientId, method);
                } else if (!patientId.equals("D123401")) {
                    output = createAppointments(host, siteCode, patientId, method);
                }

                try {
                    input.getWriter().write(output);
                } catch (IOException e) {
                    e.printStackTrace();
                }
                return input;
            }
        };

        whenHttp(server).match(custom(validBookedAppointmentRequest), method(Method.GET))
            .then(generateResponse(HttpStatus.OK_200, appointments));

        whenHttp(server).match(custom(validBookedAppointmentPostRequest), method(Method.POST))
        .then(generateResponse(HttpStatus.OK_200, appointments));

        whenHttp(server).match(custom(requestToGenerateErrorMessage), method(Method.POST))
        .then(generateResponse(HttpStatus.BAD_REQUEST_400, "Error: Unable to save appointment"));
    }

    private String createAppointments(String host, String siteCode, String patientIdentifier, Method method) {
        String appointmentTime = "09:00:00";

        String tomorrow = formatDate(xDaysFromNow(1)) + " " + appointmentTime;
        String dayAfterTomorrow = formatDate(xDaysFromNow(2)) + " " + appointmentTime;
        String next30Day = formatDate(xDaysFromNow(30)) + " " + appointmentTime;
        String next31Day = formatDate(xDaysFromNow(31)) + " " + appointmentTime;

        if(method.matchesMethod("POST")) {
            return createAppointment(host, siteCode, patientIdentifier, next31Day, "1789", "ABC Clinic", "FUTURE");
        }

        String cancelledTomorrowAppointment = createAppointment(host, siteCode, patientIdentifier, tomorrow, "1789", "ABC Clinic", "CANCELLED BY CLINIC & AUTO RE-BOOK");
        String tomorrowAppointment = createAppointment(host, siteCode, patientIdentifier, tomorrow, "1789", "ABC Clinic", "FUTURE");
        String cancelledDayAfterTomorrowAppointment = createAppointment(host, siteCode, patientIdentifier, dayAfterTomorrow, "2001", "DEF Clinic", "CANCELLED BY CLINIC");
        String dayAfterTomorrowAppointment = createAppointment(host, siteCode, patientIdentifier, dayAfterTomorrow, "2001", "DEF Clinic", "FUTURE");
        String next30DayAppointment = createAppointment(host, siteCode, patientIdentifier, next30Day, "2005", "ERG Clinic", "FUTURE");
        String next31DayAppointment = createAppointment(host, siteCode, patientIdentifier, next31Day, "3002", "XYZ Clinic", "FUTURE");

        String primaryCareAppointments = tomorrowAppointment + ',' + cancelledTomorrowAppointment + ',' + dayAfterTomorrowAppointment;
        String otherAppointments = next30DayAppointment + "," + cancelledDayAfterTomorrowAppointment + "," + next31DayAppointment;

        return String.format(appointmentsTemplate, primaryCareAppointments, otherAppointments);
    }

    private String createNotCancellableAppointments(String host, String siteCode, String patientIdentifier, Method method) {
        String appointmentTime = "09:00:00";

        String tomorrow = formatDate(xDaysFromNow(1)) + " " + appointmentTime;
        String dayAfterTomorrow = formatDate(xDaysFromNow(2)) + " " + appointmentTime;
        String next30Day = formatDate(xDaysFromNow(30)) + " " + appointmentTime;
        String next31Day = formatDate(xDaysFromNow(31)) + " " + appointmentTime;

        if(method.matchesMethod("POST")) {
            return createAppointment(host, siteCode, patientIdentifier, next31Day, "1789", "ABC Clinic", "FUTURE");
        }

        String next30DayAppointment = "";
        String next31DayAppointment = "";
        String otherAppointments = "";

        if (siteCode.equals("523")) {
            // variable length appointment
            next30DayAppointment = createAppointment(host, siteCode, patientIdentifier, 30, next30Day, "2005", "ERG-VAR Clinic", "FUTURE");
            // 10 minute appointment
            next31DayAppointment = createAppointment(host, siteCode, patientIdentifier, 10, next31Day, "3002", "XYZ-10 Clinic", "FUTURE");
            otherAppointments = next30DayAppointment + "," + next31DayAppointment;
        } else if (siteCode.equals("650")) {
            // 20 minute appointment
            next31DayAppointment = createAppointment(host, siteCode, patientIdentifier, 20, next31Day, "3002", "XYZ-20 Clinic", "FUTURE");
           otherAppointments = next31DayAppointment;
        } else {
            next31Day = formatDate(xDaysFromNow(31)) + " 10:15:00";

            // odd appointment
            next31DayAppointment = createAppointment(host, siteCode, patientIdentifier, 30, next31Day, "2005", "ERG-ODD Clinic", "FUTURE");
           otherAppointments = next31DayAppointment;
        }


        return String.format(appointmentsTemplate, "", otherAppointments);
    }

    private Date xDaysFromNow(int amount) {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DAY_OF_YEAR, amount);
        return cal.getTime();
    }

    private String createAppointment(String host, String siteCode, String patientIdentifier,
                                     int appointmentLength, String appointmentTime, String clinicId, String clinicName, String status) {
        return String.format(appointmentTemplate, host, siteCode, patientIdentifier, appointmentLength, appointmentTime, clinicId, clinicName, status);
    }

    private String createAppointment(String host, String siteCode, String patientIdentifier,
                                     String appointmentTime, String clinicId, String clinicName, String status) {
        return String.format(appointmentTemplate, host, siteCode, patientIdentifier, 30, appointmentTime, clinicId, clinicName, status);
    }
}
